<?php

class TelegramBot {
    
    private $token;
    private $apiUrl;
    private $update;
    
    public function __construct($token) {
        $this->token = $token;
        $this->apiUrl = 'https://api.telegram.org/bot' . $token;
    }
    
    /**
     * Get update from Telegram webhook
     */
    public function getUpdate() {
        $input = file_get_contents('php://input');
        $this->update = json_decode($input, true);
        return $this->update;
    }
    
    /**
     * Send message to user
     */
    public function sendMessage($chatId, $text, $parseMode = 'HTML') {
        $params = [
            'chat_id' => $chatId,
            'text' => $text,
            'parse_mode' => $parseMode
        ];
        
        return $this->makeRequest('sendMessage', $params);
    }
    
    /**
     * Send message with keyboard
     */
    public function sendMessageWithKeyboard($chatId, $text, $keyboard, $parseMode = 'HTML') {
        $params = [
            'chat_id' => $chatId,
            'text' => $text,
            'parse_mode' => $parseMode,
            'reply_markup' => json_encode($keyboard)
        ];
        
        return $this->makeRequest('sendMessage', $params);
    }
    
    /**
     * Make API request to Telegram
     */
    private function makeRequest($method, $params) {
        $url = $this->apiUrl . '/' . $method;
        
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($params));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        
        $response = curl_exec($ch);
        curl_close($ch);
        
        return json_decode($response, true);
    }
    
    /**
     * Get message text
     */
    public function getText() {
        if (isset($this->update['message'])) {
            return $this->update['message']['text'] ?? '';
        }
        return '';
    }
    
    /**
     * Get user ID
     */
    public function getUserId() {
        if (isset($this->update['message'])) {
            return $this->update['message']['from']['id'] ?? null;
        }
        return null;
    }
    
    /**
     * Get user first name
     */
    public function getUserFirstName() {
        if (isset($this->update['message'])) {
            return $this->update['message']['from']['first_name'] ?? '';
        }
        return '';
    }
    
    /**
     * Get chat ID
     */
    public function getChatId() {
        if (isset($this->update['message'])) {
            return $this->update['message']['chat']['id'] ?? null;
        }
        return null;
    }
    
    /**
     * Get chat type (private, group, supergroup, channel)
     */
    public function getChatType() {
        if (isset($this->update['message'])) {
            return $this->update['message']['chat']['type'] ?? null;
        }
        return null;
    }
    
    /**
     * Get replied message info
     */
    public function getRepliedMessage() {
        if (isset($this->update['message']['reply_to_message'])) {
            return $this->update['message']['reply_to_message'];
        }
        return null;
    }
    
    /**
     * Check if message is a reply
     */
    public function isReply() {
        return isset($this->update['message']['reply_to_message']);
    }
    
    /**
     * Get the current update data
     */
    public function getUpdate() {
        return $this->update;
    }
    
    /**
     * Log update for debugging
     */
    public function logUpdate() {
        file_put_contents(__DIR__ . '/../data/updates.log', json_encode($this->update, JSON_PRETTY_PRINT) . "\n\n", FILE_APPEND);
    }
}
